/* -- AIX/6000 System monitor 
**
**     print_dkstat.c
**
** Copyright (c) 1991-1995 Jussi Maki, All Rights Reserved.
** Copyright (c) 1993-2001 Marcel Mol, All Rights Reserved.
** NON-COMMERCIAL USE ALLOWED. YOU ARE FREE TO DISTRIBUTE
** THIS PROGRAM AND MODIFY IT AS LONG AS YOU KEEP ORIGINAL
** COPYRIGHTS.
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <curses.h>
#include "config.h"
#include "get_dkstat.h"
#include "print_dkstat.h"
#include "print.h"
#if defined(DEBUG)
#include "debug.h"
#endif

extern int show_disk_full;
extern int show_smp;
extern int sort_disks;
extern int cpu_cnt;

extern int numlines;

extern FILE *logfp;

/*
 * Variables to calculate rates
 */
float *readrate,  *writerate, *readsize,
      *writesize, *iorate,    *seeksize,
      *diskbusy;
int   *disk_sorted;

/*
 * Variables for maintaining max values, for logging mode
 */
float *readrate_max, *writerate_max,
      *readsize_max, *writesize_max,
      *iorate_max,   *seeksize_max,
      *diskbusy_max;

/*
 * Variables to determine screen location for disk figures
 */
int disk_x, disk_y;

/*
 * Local function declarations
 */
int cmp_dkstat(int *i, int *j);

/*******************************************************************
 * Functions
 *******************************************************************/

#define DKiDELTA(i,a)  ((dk1[i]->a) - (dk2[i]->a))
#define min(a,b) ((a)<(b)?(a):(b))


int
cmp_dkstat(int *i, int *j)
{
    int speed_i = readrate[*i] + writerate[*i];
    int speed_j = readrate[*j] + writerate[*j];

    if (speed_i > speed_j)
        return -1;
    else if (speed_i < speed_j)
        return 1;

    return 0;

} /* cmp_dkstat */



void
calc_dkstat(double refresh_time, struct dkstat *dk1[], struct dkstat *dk2[])
{
    int i;
    long xfers;

    for (i = 0; i < dk_cnt; i++) {
	xfers        = DKiDELTA(i, dk_xfers);
        readrate[i]  = DKiDELTA(i, dk_rblks) * dk1[i]->dk_bsize /
                       1024.0 / refresh_time;
        writerate[i] = DKiDELTA(i, dk_wblks) * dk1[i]->dk_bsize /
                       1024.0 / refresh_time;
        readsize[i]  = xfers == 0 ? 0 :
                      DKiDELTA(i, dk_rblks) * dk1[i]->dk_bsize / 1024.0 / xfers;
	writesize[i] = xfers == 0 ? 0 :
                      DKiDELTA(i, dk_wblks) * dk1[i]->dk_bsize / 1024.0 / xfers;
        iorate[i]    = xfers / refresh_time;
        seeksize[i]  = DKiDELTA(i, dk_seek) / refresh_time;
	diskbusy[i]  = DKiDELTA(i, dk_time) / refresh_time;
        if (logmode) {
            SETMAX(readrate[i],  readrate_max[i]);
            SETMAX(writerate[i], writerate_max[i]);
            SETMAX(readsize[i],  readsize_max[i]);
            SETMAX(writesize[i], writesize_max[i]);
            SETMAX(iorate[i],    iorate_max[i]);
            SETMAX(seeksize[i],  seeksize_max[i]);
            SETMAX(diskbusy[i],  diskbusy_max[i]);
        }
        if (sort_disks)
            disk_sorted[i] = i;
    }

    if (sort_disks)
        qsort((void *)disk_sorted, (size_t)dk_cnt, sizeof(int), cmp_dkstat);

    return;

} /* calc_dkstat */


void
init_dkstat_vars()
{

#if defined(DEBUG)
    OPENDB;
    fprintf(debugfp, "init_dkstat_vars: alloccing for %d disks\n", dk_cnt);
    CLOSEDB;
#endif

    if (!readrate)
        readrate = (float *) calloc(dk_cnt, sizeof(float *));
    if (!writerate)
        writerate = (float *) calloc(dk_cnt, sizeof(float *));
    if (!readsize)
        readsize = (float *) calloc(dk_cnt, sizeof(float *));
    if (!writesize)
        writesize = (float *) calloc(dk_cnt, sizeof(float *));
    if (!iorate)
        iorate = (float *) calloc(dk_cnt, sizeof(float *));
    if (!seeksize)
        seeksize = (float *) calloc(dk_cnt, sizeof(float *));
    if (!diskbusy)
        diskbusy = (float *) calloc(dk_cnt, sizeof(float *));
    if (sort_disks && !disk_sorted)
        disk_sorted = (int *) calloc(dk_cnt, sizeof(int *));

    return;

} /* init_dkstat_vars */



void
print_dkstat_init()
{

    init_dkstat_vars();
    disk_x = 0; disk_y = 13;
    move(disk_y + 0, disk_x);
    BOLDON;
    printw("IO (kB/s) read  write busy%%");
    BOLDOFF;

    return;

} /* print_dkstat_init */



void
print_dkstat(double refresh_time, struct dkstat *dk1[], struct dkstat *dk2[],
             int ndisk)
{
    int i, j;
    int row;
    
    row = disk_y + 1; 
    for (i = 0; i < (ndisk ? min(ndisk, dk_cnt) : dk_cnt) && row < numlines; i++, row++) {
	move(row, disk_x);
        j = sort_disks ? disk_sorted[i] : i;
        NAMECOLON;
	printw("%-7s", dk1[j]->diskname);
        NAMECOLOFF;
        VALUECOLON;
        
#define MB  1048576
#define KB  1024
#define dk_prw(a) if (a < 10000.0) printw(" %6.1f", a ); \
                  else if ((int) a < 10000*KB) printw(" %5dk", (int) a / KB); \
                  else printw(" %5dM", (int) a / MB);

	dk_prw(readrate[j]);
	dk_prw(writerate[j]);
	printw("  %3.0f", diskbusy[j]);

#if 0
	printw(" %6.1f %6.1f  %3.0f", readrate[j], writerate[j], diskbusy[j]);
#endif
        VALUECOLOFF;
    }

    return;

} /* printf_dkstat */



void
print_dkstat_full_init()
{

    init_dkstat_vars();
    disk_x = 0; disk_y = 5;
    if (show_smp)
        disk_y += (cpu_cnt + 2);
    move(disk_y + 0, disk_x);
    BOLDON;
    printw("DiskIO    read  write     rsize  wsize  xfers seeks blksize"
           " xrate busy");
    BOLDOFF;

    return;

} /* print_dkstat_full_init */



void
print_dkstat_full(double refresh_time,
                  struct dkstat *dk1[], struct dkstat *dk2[], int ndisk)
{
    int i, j, active;
    struct dkstat sum;
    int row;
    float xfer;

#define MB  1048576
#define KB  1024
#define dk_prw2(a) if (a < 10000.0) printw(" %6.1f", a ); \
                  else if ((int) a < 10000*KB) printw(" %5dk", (int) a / KB); \
                  else printw(" %5dM", (int) a / MB);

    active = summarize_dkstat(&sum, dk1, dk2);
    row = disk_y + 1; 
    /*
     * Per disk
     */
    for (i = 0; i < (ndisk ? min(ndisk, dk_cnt) : dk_cnt) && row < numlines - 2;
                i++, row++) {
	move(row, disk_x);
        j = sort_disks ? disk_sorted[i] : i;
	printw("%-7s", dk1[j]->diskname);
        dk_prw2(readrate[j]);
        dk_prw2(writerate[j]);
	printw(" kB/s %4.1f %4.1f kB", readsize[j], writesize[j]);
	if (iorate[j] < 1000 ) printw(" %5.1f", iorate[j]);
	else                   printw(" %5d", (int) iorate[j]);

	printw(" %5.1f %7d %4d %3.0f%%",
               seeksize[j], dk1[j]->dk_bsize, dk1[j]->dk_xrate, diskbusy[j]);
#if 0
	printw("%-7s %6.1f %6.1f kB/s %4.1f %4.1f kB %5.1f %5.1f %7d %4d"
               " %3.0f%%",
	       dk1[j]->diskname, readrate[j], writerate[j],
               readsize[j], writesize[j], iorate[j],
               seeksize[j], dk1[j]->dk_bsize, dk1[j]->dk_xrate, diskbusy[j]);
#endif
    }
    /*
     * Total
     */
    move(row, disk_x);
    printw("Total  ");
    xfer = sum.dk_rblks * sum.dk_bsize / 1024.0 / refresh_time;
    dk_prw2(xfer);
    xfer = sum.dk_wblks * sum.dk_bsize / 1024.0 / refresh_time;
    dk_prw2(xfer);
    printw(" kB/s %4.1f %4.1f kB",
           sum.dk_xfers ? sum.dk_rblks * sum.dk_bsize / 1024.0 / sum.dk_xfers
                        : 0,
           sum.dk_xfers ? sum.dk_wblks * sum.dk_bsize / 1024.0 / sum.dk_xfers
                        : 0);
    xfer = sum.dk_xfers / refresh_time; 
    if (xfer < 1000 ) printw(" %5.1f", xfer);
    else              printw(" %5d", (int) xfer);
    printw(" %5.1f %7d %4d %3.0f%%",
           sum.dk_seek  / refresh_time, sum.dk_bsize,
           sum.dk_xrate, sum.dk_time  / refresh_time);
    move(row + 1, disk_x);
    printw("Active disks: %d", active);

    return;

} /* print_dkstat_full */



void
init_log_dkstat()
{

    init_dkstat_vars();

    if (!readrate_max)
        readrate_max = (float *) calloc(dk_cnt, sizeof(float *));
    if (!writerate_max)
        writerate_max = (float *) calloc(dk_cnt, sizeof(float *));
    if (!readsize_max)
        readsize_max = (float *) calloc(dk_cnt, sizeof(float *));
    if (!writesize_max)
        writesize_max = (float *) calloc(dk_cnt, sizeof(float *));
    if (!iorate_max)
        iorate_max = (float *) calloc(dk_cnt, sizeof(float *));
    if (!seeksize_max)
        seeksize_max = (float *) calloc(dk_cnt, sizeof(float *));
    if (!diskbusy_max)
        diskbusy_max = (float *) calloc(dk_cnt, sizeof(float *));

    return;

} /* init_log_dkstat */



void
log_dkstat_header()
{

    fprintf(logfp, "# device[_max] name readrate writerate readsize writesize"
                   " iorate seeksize diskbusy xrate\n");
    fprintf(logfp, "# device    readrate     kB read / second\n");
    fprintf(logfp, "# device    writerate    kB written / second\n");
    fprintf(logfp, "# device    readsize     kB read / io\n");
    fprintf(logfp, "# device    writesize    kB written / io\n");
    fprintf(logfp, "# device    iorate       io's / second\n");
    fprintf(logfp, "# device    seeksize     cylinders seek / io\n");
    fprintf(logfp, "# device    diskbusy     busy%%\n");
    fprintf(logfp, "# device    xrate        disk max transfer rate\n");

    return;

} /* log_dkstat_header */



void
log_dkstat(double refresh_time, struct dkstat *dk1[], struct dkstat *dk2[])
{
    int i;

    for (i = 0; i < dk_cnt; i++)
	fprintf(logfp, "device %s %.1f %.1f %.1f %.1f %.1f %.1f %.0f %ld\n",
	               dk1[i]->diskname, readrate[i], writerate[i], readsize[i],
                       writesize[i], iorate[i], seeksize[i], diskbusy[i],
                       dk1[i]->dk_xrate);

    return;

} /* log_dkstat */



void
max_dkstat_init()
{
    int i;

    for (i = 0; i < dk_cnt; i++) {
        readrate_max[i] = writerate_max[i] = 0.0;
        readsize_max[i] = writesize_max[i] = 0.0;
        iorate_max[i] = seeksize_max[i] = 0.0;
        diskbusy_max[i] = 0.0;
    }

    return;

} /* max_dkstat_init */



void
log_dkstat_max(struct dkstat *dk[])
{
    int i;

    for (i = 0; dk[i]; i++)
	fprintf(logfp, "device_max %s %.1f %.1f %.1f %.1f %.1f %.1f %.0f\n",
	               dk[i]->diskname, readrate_max[i], writerate_max[i],
                       readsize_max[i], writesize_max[i],
                       iorate_max[i], seeksize_max[i], diskbusy_max[i]);

    return;

} /* log_dkstat_max */
