/* -- AIX/6000 System monitor 
**
**     monprint.c
**
** Copyright (c) 1991-1995 Jussi Maki, All Rights Reserved.
** Copyright (c) 1993-2001 Marcel Mol, All Rights Reserved.
** NON-COMMERCIAL USE ALLOWED. YOU ARE FREE TO DISTRIBUTE
** THIS PROGRAM AND MODIFY IT AS LONG AS YOU KEEP ORIGINAL
** COPYRIGHTS.
*/

#include <sys/times.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <utmp.h>
#include "config.h"
#include "monitor.h"
#include "print.h"

typedef struct {
    int nusers;
    int nactive;
    int nremote;
    time_t sleeptm;
    int nrootusers;
    int nrootactive;
    int nrootremote;
    time_t rootsleeptm;
} nusers_t;

extern char hostnm[];

extern int show_sys;
extern int show_sys_summary;
extern int show_smp;
extern int show_top;
extern int show_top_a;
extern int show_disk_full;
extern int show_inet_full;
extern int show_nfs;
#if defined(HAVE_AFS)
extern int show_afs;
#endif
extern int include_root;
extern int logtop;
extern int numcolumns;

extern FILE *logfp;

extern int aix_version;
extern int aix_release;
extern int aix_level;
extern int aix_fix;

time_t uptime;
time_t active_threshold = 60;

#ifndef UTMP_FILE
# define UTMP_FILE        "/etc/utmp"
#endif
#define	MINUTES          60
#define HOURS            (MINUTES * 60)
#define DAYS             (HOURS * 24)

/*
 * Variables to calculate rates
 */
/* sysinfo */
float cpu_sys, cpu_wait, cpu_user, cpu_idle;
float runnable, runave, swapin, swapave;
float runque_tmp, runocc_tmp, swpque_tmp, swpocc_tmp;
float lread, bread, phread, lwrite, bwrite, phwrite;
float pswitch, syscall, sysread, syswrite;
float sysfork, sysexec, rcvint, xmtint, mdmint;
float si_iget, si_namei, si_dirblk, si_readch, si_writech,
      si_rawch, si_canch, si_outch;
/* vminfo */
float pgexct, pgrclm, lockexct, backtrks, pageins, pageouts,
      pgspgins, pgspgouts, numsios, numiodone, zerofills, exfills,
      scans, cycles, pgsteals, freewts, extendwts, pendiowts,
      pings, pongs, pangs, dpongs, wpongs, cachef, cachei;
/* loadavg */
double loadv[3];
/* utmp */
nusers_t userstr;

/*
 * Variables for maintaining max values
 */
/* sysinfo */
float cpu_sys_max, cpu_wait_max, cpu_user_max, cpu_idle_max;
float runnable_max, runave_max, swapin_max, swapave_max;
float bread_max, lread_max, phread_max;
float bwrite_max, lwrite_max, phwrite_max;
float pswitch_max, syscall_max, sysread_max, syswrite_max,
      sysfork_max, sysexec_max, rcvint_max, xmtint_max, mdmint_max;
float si_iget_max, si_namei_max, si_dirblk_max, si_readch_max, si_writech_max,
      si_rawch_max, si_canch_max, si_outch_max;
/* vminfo */
float pgexct_max, pgrclm_max, lockexct_max, backtrks_max,
      pageins_max, pageouts_max, pgspgins_max, pgspgouts_max,
      numsios_max, numiodone_max, zerofills_max, exfills_max,
      scans_max, cycles_max, pgsteals_max, freewts_max, extendwts_max,
      pendiowts_max, pings_max, pongs_max, pangs_max, dpongs_max,
      wpongs_max, cachef_max, cachei_max;
/* loadavgd */
double loadv_max[3];
/* utmp */
nusers_t max_userstr;


void get_users(nusers_t *users);

/*******************************************************************
 * Functions
 *******************************************************************/

void
init_max_values(struct dkstat **dk_new, struct ifnet **ifnets_new)
{

    loadv_max[0] = loadv_max[1] = loadv_max[2] = 0.0;
    cpu_sys_max = cpu_wait_max = cpu_user_max = cpu_idle_max = 0.0;
    runnable_max = runave_max = swapin_max = swapave_max = 0.0;
    max_userstr.nusers  = 0;
    max_userstr.nactive = 0;
    max_userstr.nremote = 0;
    max_userstr.sleeptm = 0;
    max_userstr.nrootusers  = 0;
    max_userstr.nrootactive = 0;
    max_userstr.nrootremote = 0;
    max_userstr.rootsleeptm = 0;
    bread_max = lread_max = phread_max = 0.0;
    bwrite_max = lwrite_max = phwrite_max = 0.0;
    pgexct_max = pageins_max = pageouts_max = 0.0;
    pgspgins_max = pgspgouts_max = 0.0;
    pswitch_max = syscall_max = sysread_max = syswrite_max = 0.0;
    sysfork_max = sysexec_max = rcvint_max = xmtint_max = mdmint_max = 0.0;
    si_iget_max = si_namei_max = si_dirblk_max = 0.0;
    si_readch_max = si_writech_max = 0.0;
    si_rawch_max = si_canch_max = si_outch_max = 0.0;

    max_cpuinfo_init();
    max_dkstat_init();
    max_ifnet_init();
    max_nfsstat_init();
#if defined(HAVE_AFS)
    max_afsstat_init();
#endif
    if (logtop)
        max_top_init();

    return;

} /* init_max_values */


/***************************************************************************
 *                      PRINTING FUNCTIONS                                 *
 ***************************************************************************/


/*
 * generate 'len' characters 'ch' in the end of string 'str'
 */
void
strchgen(char *str, char ch, int len)
{

    while (*str)
        str++;
    if (len > 0)
        while (len--)
            *str++ = ch;
    *str = 0;

    return;

} /* strchgen */


#define BARLEN 72

#define SIDELTA(a) (si1->a - si2->a)
#define SIRATE(a)  ((si1->a - si2->a) / refresh_time)
#define VMDELTA(a) (vm1->a - vm2->a)
#define VMRATE(a)  ((vm1->a - vm2->a) / refresh_time)


void
calc_sysinfo(double refresh_time, struct sysinfo *si1, struct sysinfo *si2,
             struct vmker *vmk, struct vminfo *vm1, struct vminfo *vm2)
{
    int i;
    float cpu_sum;

    cpu_sum = (SIDELTA(cpu[CPU_IDLE])   + SIDELTA(cpu[CPU_USER]) +
               SIDELTA(cpu[CPU_KERNEL]) + SIDELTA(cpu[CPU_WAIT])) / 100.0;
    cpu_sys  = SIDELTA(cpu[CPU_KERNEL]) / cpu_sum;
    cpu_wait = SIDELTA(cpu[CPU_WAIT])   / cpu_sum;
    cpu_user = SIDELTA(cpu[CPU_USER])   / cpu_sum;
    cpu_idle = SIDELTA(cpu[CPU_IDLE])   / cpu_sum;
    SETMAX(cpu_sys,  cpu_sys_max);
    SETMAX(cpu_wait, cpu_wait_max);
    SETMAX(cpu_user, cpu_user_max);
    SETMAX(cpu_idle, cpu_idle_max);

    runocc_tmp = (float) SIDELTA(runocc);
    runque_tmp = (float) SIDELTA(runque);
    runnable   = runocc_tmp == 0.0 ? 0.0 : runque_tmp / runocc_tmp;
    runave     = runque_tmp / refresh_time;
    SETMAX(runnable, runnable_max);
    SETMAX(runave,   runave_max);

    swpocc_tmp = (float) SIDELTA(swpocc);
    swpque_tmp = (float) SIDELTA(swpque);
    swapin     = swpocc_tmp == 0.0 ? 0.0 : swpque_tmp / swpocc_tmp;
    swapave    = swpque_tmp / refresh_time;
    SETMAX(swapin,  swapin_max);
    SETMAX(swapave, swapave_max);

    bread   = (float) SIDELTA(bread);   SETMAX(bread, bread_max);
    lread   = (float) SIDELTA(lread);   SETMAX(lread, lread_max);
    phread  = (float) SIDELTA(phread);  SETMAX(phread, phread_max);
    bwrite  = (float) SIDELTA(bwrite);  SETMAX(bwrite, bwrite_max);
    lwrite  = (float) SIDELTA(lwrite);  SETMAX(lwrite, lwrite_max);
    phwrite = (float) SIDELTA(phwrite); SETMAX(phwrite, phwrite_max);

    getloadavg(loadv, 3);
    for (i = 0; i < 3; i++) {
        SETMAX(loadv[i], loadv_max[i]);
    }

    pswitch  = SIRATE(pswitch);  SETMAX(pswitch, pswitch_max);
    syscall  = SIRATE(syscall);  SETMAX(syscall, syscall_max);
    sysread  = SIRATE(sysread);  SETMAX(sysread, sysread_max);
    syswrite = SIRATE(syswrite); SETMAX(syswrite, syswrite_max);
    sysfork  = SIRATE(sysfork);  SETMAX(sysfork, sysfork_max);
    sysexec  = SIRATE(sysexec);  SETMAX(sysexec, sysexec_max);
    rcvint   = SIRATE(rcvint);   SETMAX(rcvint, rcvint_max);
    xmtint   = SIRATE(xmtint);   SETMAX(xmtint, xmtint_max);
    mdmint   = SIRATE(mdmint);   SETMAX(mdmint, mdmint_max);

    si_iget    = SIRATE(iget);    SETMAX(si_iget, si_iget_max);
    si_namei   = SIRATE(namei);   SETMAX(si_namei, si_namei_max);
    si_dirblk  = SIRATE(dirblk);  SETMAX(si_dirblk, si_dirblk_max);
    si_readch  = SIRATE(readch);  SETMAX(si_readch, si_readch_max);
    si_writech = SIRATE(writech); SETMAX(si_writech, si_writech_max);
    si_rawch   = SIRATE(rawch);   SETMAX(si_rawch, si_rawch_max);
    si_canch   = SIRATE(canch);   SETMAX(si_canch, si_canch_max);
    si_outch   = SIRATE(outch);   SETMAX(si_outch, si_outch_max);

    pgexct    = VMRATE(pgexct);    SETMAX(pgexct,    pgexct_max);
    pgrclm    = VMRATE(pgrclm);    SETMAX(pgrclm,    pgrclm_max);
    lockexct  = VMRATE(lockexct);  SETMAX(lockexct,  lockexct_max);
    backtrks  = VMRATE(backtrks);  SETMAX(backtrks,  backtrks_max);
    pageins   = VMRATE(pageins);   SETMAX(pageins,   pageins_max);
    pageouts  = VMRATE(pageouts);  SETMAX(pageouts,  pageouts_max);
    pgspgins  = VMRATE(pgspgins);  SETMAX(pgspgins,  pgspgins_max);
    pgspgouts = VMRATE(pgspgouts); SETMAX(pgspgouts, pgspgouts_max);
    numsios   = VMRATE(numsios);   SETMAX(numsios,   numsios_max);
    numiodone = VMRATE(numiodone); SETMAX(numiodone, numiodone_max);
    zerofills = VMRATE(zerofills); SETMAX(zerofills, zerofills_max);
    exfills   = VMRATE(exfills);   SETMAX(exfills,   exfills_max);
    scans     = VMRATE(scans);     SETMAX(scans,     scans_max);
    cycles    = VMRATE(cycles);    SETMAX(cycles,    cycles_max);
    pgsteals  = VMRATE(pgsteals);  SETMAX(pgsteals,  pgsteals_max);
    freewts   = VMRATE(freewts);   SETMAX(freewts,   freewts_max);
    extendwts = VMRATE(extendwts); SETMAX(extendwts, extendwts_max);
    pendiowts = VMRATE(pendiowts); SETMAX(pendiowts, pendiowts_max);
    pings     = VMRATE(pings);     SETMAX(pings,     pings_max);
    pongs     = VMRATE(pongs);     SETMAX(pongs,     pongs_max);
    pangs     = VMRATE(pangs);     SETMAX(pangs,     pangs_max);
    dpongs    = VMRATE(dpongs);    SETMAX(dpongs,    dpongs_max);
    wpongs    = VMRATE(wpongs);    SETMAX(wpongs,    wpongs_max);
    cachef    = VMRATE(cachef);    SETMAX(cachef,    cachef_max);
    cachei    = VMRATE(cachei);    SETMAX(cachei,    cachei_max);

    return;

} /* calc_sysinfo */



void
calc_users()
{

    get_users(&userstr);

    SETMAX(userstr.nusers,      max_userstr.nusers);
    SETMAX(userstr.nactive,     max_userstr.nactive);
    SETMAX(userstr.nremote,     max_userstr.nremote);
    SETMAX(userstr.sleeptm,     max_userstr.sleeptm);
    SETMAX(userstr.nrootusers,  max_userstr.nrootusers);
    SETMAX(userstr.nrootactive, max_userstr.nrootactive);
    SETMAX(userstr.nrootremote, max_userstr.nrootremote);
    SETMAX(userstr.rootsleeptm, max_userstr.rootsleeptm);

    return;

} /* calc_users */



void get_users(nusers_t *users)
{
    FILE *utmpf;
    int nusers, nremote, nactive;
    int nrootusers, nrootremote, nrootactive;
    int root;
    struct utmp utmp;
    struct stat stat_buf;
    char device[12+5+1] = "/dev/";
    time_t now, active_threshold = 60;
    time_t last_access, last_change, last_mod, inact, rootinact;
  
    if (!(utmpf = fopen(UTMP_FILE, "r"))) {
        perror(UTMP_FILE);
        users->nusers = -1;
        return;
    }

    nusers        = 0; /* total session count */
    nremote       = 0; /* total remote (network) sessions */
    nactive       = 0; /* total active (see active_threshold) sessions */
    inact         = 0; /* total inactive seconds */
    nrootusers    = 0; /* total root sessions */
    nrootremote   = 0; /* total root remote sessions */
    nrootactive   = 0; /* total root active sessions */
    rootinact     = 0; /* total root inactive seconds */
    now = time(0);
  
    while (fread((char *)&utmp, sizeof(utmp), 1, utmpf) == 1)
        if (*utmp.ut_name && utmp.ut_type == USER_PROCESS) {
            nusers++;
            root = strcmp(utmp.ut_name, "root") ? 0 : 1;
            if (root)
                nrootusers++;
            memset(device+5, '\0', 12+1);
            strcat(device+5, utmp.ut_line);
            if (stat(device, &stat_buf)) {
                last_access = -1;
                last_mod    = -1;
                last_change = -1;
            }
            else {
	        if (*utmp.ut_host) {
                    nremote++;
                    if (root)
                        nrootremote++;
                }
                /* when data is typed at terminal, all times change */
                /* when inode is changed */
                last_access = now - stat_buf.st_atime;
                /* when inode, or data is changed */
                last_mod    = now - stat_buf.st_mtime;
                /* when data of file changes */
                last_change = now - stat_buf.st_ctime;
                if (last_mod < active_threshold) {
                    nactive++;
                    if (root)
                        nrootactive++;
                }
                inact += last_mod;
                if (root)
                    rootinact += last_mod;
            }
        }
        else if (utmp.ut_type == BOOT_TIME)
            uptime = now - utmp.ut_time;

    fclose(utmpf);

    users->nusers        = nusers;
    users->nactive       = nactive;
    users->nremote       = nremote;
    users->nrootusers    = nrootusers;
    users->nrootactive   = nrootactive;
    users->nrootremote   = nrootremote;
    users->sleeptm       = inact;
    users->rootsleeptm   = rootinact;

    return;

} /* get_users */



void
print_sysinfo_init()
{
    int x, y;

    NAMECOLON;
    move( 0,  0); printw("%s: %s", MONITOR_NAME, hostnm);
    NAMECOLOFF;
    BOLDON;
    move( 1,  0); printw("Uptime:");
    move( 1, 26); printw("Users:");
    move( 2,  0); printw("CPU:");
    move( 2, 50); printw("Refresh:");
    BOLDOFF;
    NAMECOLON;
    move( 3,  0); printw("0%%             25%%              50%%"
                         "               75%%              100%%");
    move( 6,  0); printw("Runnable (Swap-in) processes");
    move( 6, 43); printw("load average:");
    NAMECOLOFF;
    BOLDON;
    move( 8,  0); printw("Memory    Real     Virtual");
    BOLDOFF;
    NAMECOLON;
    move( 9,  0); printw("free");
    move(10,  0); printw("procs");
    move(11,  0); printw("files");
    move(12,  0); printw("total");
    NAMECOLOFF;

    x = 30; y = 8;
    BOLDON;
    move(y + 0, x); printw("Paging (4kB)");
    BOLDOFF;
    NAMECOLON;
    x += 7;
    move(y + 1, x); printw("pgfaults");
    move(y + 2, x); printw("pgin");
    move(y + 3, x); printw("pgout");
    move(y + 4, x); printw("pgsin");
    move(y + 5, x); printw("pgsout");
    NAMECOLOFF;

    x = 46; y = 8;
    BOLDON;
    move(y + 0, x); printw("Process events");
    BOLDOFF;
    NAMECOLON;
    x += 8;
    move(y + 1, x); printw("pswitch");
    move(y + 2, x); printw("syscall");
    move(y + 3, x); printw("read");
    move(y + 4, x); printw("write");
    move(y + 5, x); printw("fork");
    move(y + 6, x); printw("exec");
    move(y + 7, x); printw("rcvint");
    move(y + 8, x); printw("xmtint");
#ifdef mon_notdef
    move(y + 9, x); printw("mdmint");
#endif
    NAMECOLOFF;
    
    x = 62; y = 8;
    BOLDON;
    move(y + 0, x); printw("   File/TTY-IO");
    BOLDOFF;
    NAMECOLON;
    x += 8;
    move(y + 1, x); printw("iget");
    move(y + 2, x); printw("namei");
    move(y + 3, x); printw("dirblk");
    move(y + 4, x); printw("readch");
    move(y + 5, x); printw("writech");
    move(y + 6, x); printw("ttyrawch");
    move(y + 7, x); printw("ttycanch");
    move(y + 8, x); printw("ttyoutch");
    NAMECOLOFF;

    return;

} /* print_sysinfo_init */



void
print_sysinfo(double refresh_time, time_t now,
              struct sysinfo *si1, struct sysinfo *si2,
              struct vmker *vmk, struct vminfo *vm1, struct vminfo *vm2)
{
    char   bar[BARLEN];
    char  *datestr;
    int    x, y;
    struct tms tbuf;
    int    days, hours, minutes, users;

    datestr = ctime(&now);
    move(0, numcolumns - strlen(datestr)); printw(datestr);

    /*
     * logged on users
     * Note: calc_users also gets boot time from utmp so can calculate uptime
     */
    calc_users();
    if (!uptime)
        uptime = times(&tbuf) / HZ;
    uptime -= (days = uptime / DAYS) * DAYS;
    uptime -= (hours = uptime / HOURS) * HOURS;
    uptime -= (minutes = uptime / MINUTES) * MINUTES;
    move(1, 8); printw("%3d days, %02d:%02d",
	                  days, hours, minutes);
    move(1, 33);
    uptime = userstr.sleeptm - (include_root ? 0 : userstr.rootsleeptm);
    users = userstr.nusers  - (include_root ? 0 : userstr.nrootusers),
    printw("%3d of %3d active %d remote %02d:%02d sleep time",
	    userstr.nactive - (include_root ? 0 : userstr.nrootactive),
            users,
            userstr.nremote - (include_root ? 0 : userstr.nrootremote),
            users ? uptime / 60 / users : 0,
            users ? uptime % 60 / users : 0);


    move(2, 59); printw("%5.2f s", refresh_time);

    move(2, 5);
    printw("User %4.1lf%% Sys %4.1lf%% Wait %4.1lf%% Idle %4.1lf%%",
            cpu_user, cpu_sys, cpu_wait, cpu_idle);
    bar[0] = 0;
    strchgen(bar, '>', (int) (cpu_user * BARLEN/100));
    strchgen(bar, '=', (int) (cpu_sys  * BARLEN/100));
    strchgen(bar, '-', (int) (cpu_wait * BARLEN/100));
    /* strchgen(bar, '.', (int) (cpu_idle * BARLEN/100)); */
    move(4, 0); printw("%s", bar);
    clrtoeol();

    move(6, 29); printw("%5.2lf (%4.2f)", runave, swapave);

    move(6, 57); printw("%5.2lf, %5.2lf, %5.2lf", loadv[0], loadv[1], loadv[2]);

    move(9,  7); printw("%6d MB %6d MB", vmk->freemem   * 4 / 1024,
                                         vmk->freevmem  * 4 / 1024);
    move(10,  7); printw("%6d MB %6d MB", 
	                (vmk->totalmem - vmk->numperm - vmk->freemem) *
                         4 / 1024,
	                (vmk->totalvmem - vmk->freevmem) * 4 / 1024);
    move(11, 7); printw("%6d MB",        vmk->numperm   * 4 / 1024);
    move(12, 7); printw("%6d MB %6d MB", vmk->totalmem  * 4 / 1024,
                                         vmk->totalvmem * 4 / 1024);

    x = 30; y = 8;
    move(y + 1, x); printw("%6.1f", pgexct);
    move(y + 2, x); printw("%6.1f", pageins);
    move(y + 3, x); printw("%6.1f", pageouts);
    move(y + 4, x); printw("%6.1f", pgspgins);
    move(y + 5, x); printw("%6.1f", pgspgouts);

#define myprw1(a) if (a < 100000) printw("%5d", (int) a ); \
                  else printw("%4dk", (int) a / 1000);

    x = 48; y = 8;
    move(y + 1, x); myprw1(pswitch);
    move(y + 2, x); myprw1(syscall);
    move(y + 3, x); myprw1(sysread);
    move(y + 4, x); myprw1(syswrite);
    move(y + 5, x); myprw1(sysfork);
    move(y + 6, x); myprw1(sysexec);
    move(y + 7, x); myprw1(rcvint);
    move(y + 8, x); myprw1(xmtint);
#ifdef mon_notdef
    move(y + 9, x); myprw1(mdmint);
#endif
    
#define MB  1048576
#define KB  1024
#define myprw2(a) if (a < 100000) printw("%7d", (int) a ); \
                  else if (a < 100000*KB) printw("%6dk", (int) a / KB); \
                  else printw("%6dM", (int) a / MB);

    x = 62; y = 8;
    move(y + 1, x); printw("%7d", (int) si_iget);
    move(y + 2, x); printw("%7d", (int) si_namei);
    move(y + 3, x); printw("%7d", (int) si_dirblk);
    move(y + 4, x); myprw2(si_readch);
    move(y + 5, x); myprw2(si_writech);
    move(y + 6, x); printw("%7d", (int) si_rawch);
    move(y + 7, x); printw("%7d", (int) si_canch);
    move(y + 8, x); printw("%7d", (int) si_outch);
    return;

} /* print_sysinfo */



void
print_summary_init()
{

    BOLDON;
    move(0, 35); printw("%s", hostnm);
    move(0,  0); printw("Load averages:");
    move(1,  60); printw("For non-commercial");
    move(2,  66); printw("use only!");
    move(1,  0); printw("Cpu states:");
    move(2,  0); printw("Logged on:");
    move(3,  0); printw("Real memory:");
    move(4,  0); printw("Virtual memory:");
    BOLDOFF;

    return;

} /* print_summary_init */



void
print_summary(double refresh_time, time_t now,
              struct sysinfo *si1, struct sysinfo *si2,
              struct vmker *vmk, struct vminfo *vm1, struct vminfo *vm2)
{
    time_t sleep;
    int users;
    char *datestr;

    /*
     * loadavg date-time
     */

    move(0, 15); printw("%5.2lf, %5.2lf, %5.2lf", loadv[0], loadv[1], loadv[2]);

    datestr = ctime(&now);
    move(0, numcolumns - strlen(datestr)); printw(datestr);

    /*
     * cpu-states
     */
    move(1, 12); printw("%4.1f%% user %4.1f%% system "
                        "%4.1f%% wait %4.1f%% idle  ",
	                cpu_user, cpu_sys, cpu_wait, cpu_idle);
    /*
     * logged on users
     */
    calc_users();
    move(2, 11);
    sleep = userstr.sleeptm - (include_root ? 0 : userstr.rootsleeptm);
    users = userstr.nusers  - (include_root ? 0 : userstr.nrootusers),
    printw("%3d users %3d active %d remote %02d:%02d sleep time",
            users,
	    userstr.nactive - (include_root ? 0 : userstr.nrootactive),
            userstr.nremote - (include_root ? 0 : userstr.nrootremote),
            users ? sleep / 60 / users : 0,
            users ? sleep % 60 / users : 0);

    /*
     * memory-states
     */
    move(3, 13); printw("%6.1fM procs %6.1fM files %6.1fM free %6.1fM total",
	                (vmk->totalmem - vmk->numperm - vmk->freemem) *
                         4 / 1024.0,
                        vmk->numperm * 4 / 1024.0,
                        vmk->freemem * 4 / 1024.0,
	                vmk->totalmem * 4 / 1024.0);
    move(4, 27); printw("%6.1fM used  %6.1fM free %6.1fM total",
	                (vmk->totalvmem - vmk->freevmem) * 4 / 1024.0,
                        vmk->freevmem  * 4 / 1024.0,
	                vmk->totalvmem * 4 / 1024.0);
    return;

} /* print_summary */



void
print_log_header()
{

    fprintf(logfp, "# Monitor -- %s\n", MONITOR_NAME);
    fprintf(logfp, "#            compiled on AIX %d.%d.%d.%d running on AIX %d.%d.%d.%d\n",
          AIX_VERSION, AIX_RELEASE, AIX_LEVEL, AIX_FIX,
          aix_version, aix_release, aix_level, aix_fix);
    fprintf(logfp, "#interval/sample hostname interval_len startime "
                   "endtime dayname day month year monthname weekdaynum"
                   " weeknum, sunday is day 0\n");
    fprintf(logfp, "# cpubusy[_max] sys%% wait%% user%% idle%%\n");
    fprintf(logfp, "# runque[_max] runnable/sample runnable/sec runcount runocccount\n");
    fprintf(logfp, "# users[_max] number_of_users active_users network_remote"
                   " total_inactive_secs number_of_rootusers active_rootusers"
                   " network_rootremote total_rootinactive_secs\n");
    fprintf(logfp, "# uptime secs_uptime\n");
    fprintf(logfp, "# swapin[_max] swapin/sample swapin/sec\n");
    fprintf(logfp, "# loadaverage[_max] last_min last_5_mins last_quarter\n");

    fprintf(logfp, "# realmem total_MB free_MB bad_MB files_MB max_files_MB"
                   " client_MB max_client_MB non_pinnable_MB\n");
    fprintf(logfp, "# virtualmem total_MB free_MB reserved_MB\n");
/* since we now only extract known fields in our own structure do
   not save this anymore...
    fprintf(logfp, "# vmker n0 n1 n2 n3 n4 n5 n6 n7 n16 n18 n19\n");
*/

    fprintf(logfp, "# vmpaging[_max] pgexct pgrclm lockexct backtrks pageins"
                   " pageouts pgspgins pgspgouts\n");
    fprintf(logfp, "# vmpaging pgexct    cnt of page faults\n");
    fprintf(logfp, "# vmpaging pgrclm    cnt of page reclaims\n");
    fprintf(logfp, "# vmpaging lockexct  cnt of lockmisses\n");
    fprintf(logfp, "# vmpaging backtrks  cnt of backtracks\n");
    fprintf(logfp, "# vmpaging pageins   cnt of pages paged in\n");
    fprintf(logfp, "# vmpaging pageouts  cnt of pages paged out\n");
    fprintf(logfp, "# vmpaging pgspgins  cnt of page ins from paging space\n");
    fprintf(logfp, "# vmpaging pgspgouts cnt of page outs from paging space\n");

    fprintf(logfp, "# vmios[_max] numsios numiodone zerofills exfills scans"
                   " cycles pgsteals freewts extendwts pendiowts\n");
    fprintf(logfp, "# vmios numsios   cnt of start I/Os\n");
    fprintf(logfp, "# vmios numiodone cnt of iodone\n");
    fprintf(logfp, "# vmios zerofills cnt of zero filled pages\n");
    fprintf(logfp, "# vmios exfills   cnt of exec filled pages\n");
    fprintf(logfp, "# vmios scans     cnt of page scans by clock\n");
    fprintf(logfp, "# vmios cycles    cnt of clock hand cycles\n");
    fprintf(logfp, "# vmios pgsteals  cnt of page steals\n");
    fprintf(logfp, "# vmios freewts   cnt of free frame waits\n");
    fprintf(logfp, "# vmios extendwts cnt of extend XPT waits\n");
    fprintf(logfp, "# vmios pendiowts cnt of pending I/O waits\n");

    fprintf(logfp, "# vmpongs[_max] pings pongs pangs dpongs wpongs cachef"
                   " cachei\n");
    fprintf(logfp, "# vmpongs pings  cnt of ping-pongs: source => alias\n");
    fprintf(logfp, "# vmpongs pongs  cnt of ping-pongs: alias => source\n");
    fprintf(logfp, "# vmpongs pangs  cnt of ping-pongs: alias => alias\n");
    fprintf(logfp, "# vmpongs dpongs cnt of ping-pongs: alias page delete\n");
    fprintf(logfp, "# vmpongs wpongs cnt of ping-pongs: alias page writes\n");
    fprintf(logfp, "# vmpongs cachef cnt of ping-pong cache flushes\n");
    fprintf(logfp, "# vmpongs cachei cnt of ping-pong cache invalidates\n");

    fprintf(logfp, "# pswitch[_max] process_switches/second\n");
    fprintf(logfp, "# syscall[_max] total read write fork exec\n");
    fprintf(logfp, "# syscall    total total_system_calls/second\n");
    fprintf(logfp, "# syscall    read  read_syscalls/second\n");
    fprintf(logfp, "# syscall    write write_syscalls/second\n");
    fprintf(logfp, "# syscall    fork  fork_syscalls/second\n");
    fprintf(logfp, "# syscall    exec  exec_syscalls/second\n");
    fprintf(logfp, "# interrupts[_max] rcvint xmtint mdmint\n");
    fprintf(logfp, "# interrupts rcvint tty_receive_interrupts/second\n");
    fprintf(logfp, "# interrupts xmtint tty_transmit_interrupts/second\n");
    fprintf(logfp, "# interrupts mdmint modem_interrupts/second\n");

    fprintf(logfp, "# files[_max] iget namei dirblk\n");
    fprintf(logfp, "# files    iget   inode_lookups/second\n");
    fprintf(logfp, "# files    namei  name_to_inode_conversions/second\n");
    fprintf(logfp, "# files    dirblk 512_byte_directoryblocks_read/second\n");
    fprintf(logfp, "# chars[_max] readch writech ttyrawch ttycanch ttyoutch\n");
    fprintf(logfp, "# chars  readch  characters_read_by_read_syscall/second\n");
    fprintf(logfp, "# chars  writech"
                   " characters_written_by_write_syscall/second\n");
    fprintf(logfp, "# chars  ttyrawch tty_raw_characters_in/second\n");
    fprintf(logfp, "# chars  ttycanch tty_canonical_characters_in/second\n");
    fprintf(logfp, "# chars  ttyoutch tty_characters_out/second\n");

    log_cpuinfo_header();
    log_dkstat_header();
    log_ifnet_header();
    log_nfsstat_header();
#if defined(HAVE_AFS)
    log_afsstat_header();
#endif
    if (logtop)
        log_top_header();
    print_fs_header();

    return;

} /* print_log_header */



void
print_log(double refresh_time, struct sysinfo *si1, struct sysinfo *si2,
          struct vmker *vmk, struct vminfo *vm1, struct vminfo *vm2,
          struct cpuinfo *cpus1, struct cpuinfo *cpus2,
          struct dkstat *dk1[], struct dkstat *dk2[],
          struct ifnet *if1[], struct ifnet *if2[],
          t_nfs_stats *nfs1, t_nfs_stats *nfs2,
#if defined(HAVE_AFS)
          afsstat_t *afs1, afsstat_t *afs2,
#endif
          topcpu_t *top, int nprocs)
{
    struct tms tbuf;

    calc_sysinfo(refresh_time, si1, si2, vmk, vm1, vm2);
    calc_users();

    /*
     * CPU busy figures
     */
    fprintf(logfp, "cpubusy %.1f %.1f %.1f %.1f\n",
                cpu_sys, cpu_wait, cpu_user, cpu_idle);

    /*
     * System run and swap load
     */
    fprintf(logfp, "runque %.2f %.2f %.2f %.2f\n",
                   runnable, runave, runque_tmp, runocc_tmp);

    fprintf(logfp, "swapin %.2f %.2f\n", swapin, swapave);

    fprintf(logfp, "users %d %d %d %ld %d %d %d %ld\n", userstr.nusers,
                   userstr.nactive, userstr.nremote, userstr.sleeptm,
                   userstr.nrootusers, userstr.nrootactive,
                   userstr.nrootremote, userstr.rootsleeptm);
    fprintf(logfp, "uptime %ld\n", times(&tbuf)/HZ);

    fprintf(logfp, "devread %.2f %.2f %.2f\n", bread, lread, phread);
    fprintf(logfp, "devwrite %.2f %.2f %.2f\n", bwrite, lwrite, phwrite);

    fprintf(logfp, "loadaverage %.2f %.2f %.2f\n",
                   loadv[0], loadv[1], loadv[2]);

    /*
     * Memory info
     */
    fprintf(logfp, "realmem %d %d %d %d %d %d %d %d\n",
                   vmk->totalmem * 4 / 1024,
	           vmk->freemem * 4 / 1024, vmk->badmem * 4 / 1024,
                   vmk->numperm * 4 / 1024, vmk->maxperm * 4 / 1024,
                   vmk->numclient * 4 / 1024, vmk->maxclient * 4 / 1024,
                   vmk->nonpinnable * 4 / 1024);
    fprintf(logfp, "virtualmem %d %d %d\n",
                   vmk->totalvmem * 4 / 1024, vmk->freevmem * 4 / 1024,
                   vmk->pageresv * 4 / 1024);
/* since we now only extract known fields in our own structure do
   not save this anymore...
    fprintf(logfp, "vmker %d %d %d %d %d %d %d %d %d %d %d\n",
                   vmk->n0, vmk->n1, vmk->n2, vmk->n3, vmk->n4, vmk->n5,
                   vmk->n6, vmk->n7, vmk->n16, vmk->n18, vmk->n19);
*/
    fprintf(logfp, "vmpaging %.1f %.1f %.1f %.1f %.1f %.1f %.1f %.1f\n",
                 pgexct, pgrclm, lockexct, backtrks, pageins, pageouts,
                 pgspgins, pgspgouts);
    fprintf(logfp, "vmios %.1f %.1f %.1f %.1f %.1f %.1f %.1f %.1f %.1f %.1f\n",
                 numsios, numiodone, zerofills, exfills,
                 scans, cycles, pgsteals, freewts, extendwts, pendiowts);
    fprintf(logfp, "vmpongs %.1f %.1f %.1f %.1f %.1f %.1f %.1f\n",
                 pings, pongs, pangs, dpongs, wpongs, cachef, cachei);

    /*
     * System call info
     */
    fprintf(logfp, "pswitch %.1f\n", pswitch);
    fprintf(logfp, "syscall %.1f %.1f %.1f %.1f %.1f\n",
                   syscall, sysread, syswrite, sysfork, sysexec);
    fprintf(logfp, "interrupts %.1f %.1f %.1f\n",  rcvint, xmtint, mdmint);

    fprintf(logfp, "files %.1f %.1f %.1f\n", si_iget, si_namei, si_dirblk);
    fprintf(logfp, "chars %.1f %.1f %.1f %.1f %.1f\n", si_readch,
                   si_writech, si_rawch, si_canch, si_outch);
    
    calc_cpuinfo(refresh_time, cpus1, cpus2);
    log_cpuinfo();
    calc_dkstat(refresh_time, dk1, dk2);
    log_dkstat(refresh_time, dk1, dk2);
    calc_ifnet(refresh_time, if1, if2);
    log_ifnet(refresh_time, if1, if2);
    calc_nfsstat(refresh_time, nfs1, nfs2);
    log_nfsstat(refresh_time, nfs1, nfs2);
#if defined(HAVE_AFS)
    calc_afsstat(refresh_time, afs1, afs2);
    log_afsstat(refresh_time, afs1, afs2);
#endif
    if (logtop) {
        calc_top(refresh_time, top, nprocs);
        log_top(refresh_time, top, nprocs);
    }

    fflush(logfp);

    return;

} /* print_log */



void
print_log_max(struct dkstat *dk1[], struct ifnet *if1[])
{

    /*
     * CPU busy figures
     */
    fprintf(logfp, "cpubusy_max %.1f %.1f %.1f %.1f\n",
                cpu_sys_max, cpu_wait_max, cpu_user_max, cpu_idle_max);

    /*
     * System run and swap load
     */
    fprintf(logfp, "runque_max %.2f %.2f\n", runnable_max, runave_max);

    fprintf(logfp, "swapin_max %.2f %.2f\n", swapin_max, swapave_max);
    fprintf(logfp, "users_max %d %d %d %ld %d %d %d %ld\n", max_userstr.nusers,
                   max_userstr.nactive, max_userstr.nremote,
                   max_userstr.sleeptm,
                   max_userstr.nrootusers, max_userstr.nrootactive,
                   max_userstr.nrootremote, max_userstr.rootsleeptm);

    fprintf(logfp, "loadaverage_max %.2f %.2f %.2f\n",
                   loadv_max[0], loadv_max[1], loadv_max[2]);

    fprintf(logfp, "vmpaging_max %.1f %.1f %.1f %.1f %.1f %.1f %.1f %.1f\n",
                 pgexct_max, pgrclm_max, lockexct_max, backtrks_max,
                 pageins_max, pageouts_max, pgspgins_max, pgspgouts_max);
    fprintf(logfp, "vmios_max %.1f %.1f %.1f %.1f %.1f %.1f %.1f %.1f %.1f"
                   " %.1f\n",
                 numsios_max, numiodone_max, zerofills_max, exfills_max,
                 scans_max, cycles_max, pgsteals_max, freewts_max,
                 extendwts_max, pendiowts_max);
    fprintf(logfp, "vmpongs_max %.1f %.1f %.1f %.1f %.1f %.1f %.1f\n",
                 pings_max, pongs_max, pangs_max, dpongs_max, wpongs_max,
                 cachef_max, cachei_max);

    /*
     * System call info
     */
    fprintf(logfp, "pswitch_max %.1f\n", pswitch_max);
    fprintf(logfp, "syscalls_max %.1f %.1f %.1f %.1f %.1f\n", syscall_max,
                   sysread_max, syswrite_max, sysfork_max, sysexec_max);
    fprintf(logfp, "interrupts_max %.1f %.1f %.1f\n",  rcvint_max,
                   xmtint_max, mdmint_max);

    fprintf(logfp, "files_max %.1f %.1f %.1f\n",
                   si_iget_max, si_namei_max, si_dirblk_max);
    fprintf(logfp, "chars_max %.1f %.1f %.1f %.1f %.1f\n", si_readch_max,
                   si_writech_max, si_rawch_max, si_canch_max, si_outch_max);
    
    log_cpuinfo_max();
    log_dkstat_max(dk1);
    log_ifnet_max(if1);
    log_nfsstat_max();
#if defined(HAVE_AFS)
    log_afsstat_max();
#endif
    if (logtop)
        log_top_max();

    return;

} /* print_log_max */



/*
 * Note: initscreen does things based on show_xxx variables.
 *       someone else is responsible that no incompatible show_xxx vars
 *       are turned on at the same time...
 */
void
initscreen()
{
    static have_mesa = 0;

    clear();

    NAMECOLON;
    if (show_sys_summary) {
        print_summary_init();
        if (show_smp)
            print_cpuinfo_init();
        if (show_disk_full)
            print_dkstat_full_init();
        if (show_inet_full)
            print_ifnet_init();
    }
    if (show_sys) {
        print_sysinfo_init();
        print_ifnet_init();
        print_dkstat_init();
        if (show_nfs)
            print_nfsstat_init();
#if defined(HAVE_AFS)
        if (show_afs)
            print_afsstat_init();
#endif
        if (!have_mesa) {
            move(19,0); printw("MESA Consulting");
            move(20,0); printw("Unix and Internet");
            move(21,0); printw("Technology Specialists");
            move(22,0); printw("http://www.mesa.nl");
            have_mesa++;
	}
        BOLDON;
        move(7,8); printw("FOR NON-COMMERCIAL USE ONLY     see http://www.mesa.nl/monitor");
        BOLDOFF;
    }
/*  nothing to do for top at this time ...
    if (show_top || show_top_a)
        top_print_init();
*/
    move(0, 0);
    refresh();

    return;

} /* initscreen */
