/* -- AIX/6000 System monitor 
**
**     get_top.c
**
** Copyright (c) 1991-1995 Jussi Maki, All Rights Reserved.
** Copyright (c) 1993-2001 Marcel Mol, All Rights Reserved.
** NON-COMMERCIAL USE ALLOWED. YOU ARE FREE TO DISTRIBUTE
** THIS PROGRAM AND MODIFY IT AS LONG AS YOU KEEP ORIGINAL
** COPYRIGHTS.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <procinfo.h>
#include "config.h"
#include "get_top.h"
#include "get_topP.h"
#include "getproc.h"

int nproc1 = 0, nproc2 = 0;
struct procinfo proc1[NPROCS];
struct procinfo proc2[NPROCS];
struct userinfo user1[NPROCS];
struct userinfo user2[NPROCS];
struct procsortinfo top_sortinfo[NPROCS];
int *nproc_old, *nproc_cur, *nproc_save;
struct procinfo *proc_old, *proc_cur, *proc_save;
struct userinfo *user_old, *user_cur, *user_save;

int get_topcpu(topcpu_t *top, int ntops)
{
    double cpusum;
    static int initted = 0;
    int nprocs;

    if (!initted) {
        proc_old  = proc1;    user_old  = user1;    nproc_old = &nproc1;
        proc_cur  = proc2;    user_cur  = user2;    nproc_cur = &nproc2;
        initted = 1;
    }
    nprocs = *nproc_cur = top_getprocinfo(proc_cur, user_cur);
    cpusum = top_calcsortinfo(proc_old, user_old, *nproc_old, 
			      proc_cur, user_cur, *nproc_cur, top_sortinfo);
    get_topdata(top, ntops, proc_cur, user_cur, top_sortinfo, 
	        *nproc_cur, cpusum);
    swap_ptrs(proc_cur,  proc_old,  proc_save);
    swap_ptrs(nproc_cur, nproc_old, nproc_save);
    swap_ptrs(user_cur,  user_old,  user_save);

    return nprocs;

} /*get_topcpu */



void get_topdata(topcpu_t *top, int ntops, struct procinfo *proc,
		 struct userinfo *user, struct procsortinfo *procsortinfo,
		 int nproc, double cpusum)
{
    int i, j;

    for (j = 0; j < min(ntops, nproc); j++) {
        i = procsortinfo[j].index;
        if (proc[i].pi_stat == SZOMB) {
            strcpy(top[j].progname, "<defunct>");
        }
        else if (proc[i].pi_flag & SKPROC) {
            char buf[256];
            strcpy(buf, "Kernel (");
            strcat(buf, user[i].ui_comm);
            strcat(buf, ")");
            strncpy(top[j].progname, buf, TOPCPU_PROGNAME_LEN-1);
        }
        else {
            strncpy(top[j].progname, user[i].ui_comm, TOPCPU_PROGNAME_LEN-1);
        }
        top[j].progname[TOPCPU_PROGNAME_LEN-1]='\0';
        top[j].pid         = proc[i].pi_pid;
        top[j].ppid        = proc[i].pi_ppid;
        top[j].uid         = proc[i].pi_uid;
        top[j].pri         = proc[i].pi_pri;
        top[j].nice        = proc[i].pi_nice;
        top[j].stat        = proc[i].pi_stat;
        top[j].memsize_1k  = procsortinfo[j].vir;
        top[j].ressize_1k  = procsortinfo[j].res;
        top[j].pageflt     = user[i].ui_ru.ru_minflt;
        top[j].starttime   = user[i].ui_start;
        top[j].cpu_utime   = user[i].ui_ru.ru_utime.tv_sec +
                             (double)user[i].ui_ru.ru_utime.tv_usec/1000000.0;
        top[j].cpu_stime   = user[i].ui_ru.ru_stime.tv_sec +
                             (double)user[i].ui_ru.ru_stime.tv_usec/1000000.0;
        top[j].tty         = user[i].ui_ttyp;

        top[j].deltapageflt= procsortinfo[j].deltapageflt;
        /* cpusum should be equal to refresh_time */
        top[j].cputime_prs = ((double)procsortinfo[j].deltacputime /
			      (double)cpusum * 1000.0);
    }

    return;

} /* get_topdata */



/*
 *  Compare routines for qsort()
 */
int cmp_cpu(struct procsortinfo *a, struct procsortinfo *b)
{

    return (a->deltacputime > b->deltacputime) ? -1 :
           (a->deltacputime < b->deltacputime) ?  1 :
           (a->cputime > b->cputime)           ? -1 :
           (a->cputime < b->cputime)           ?  1 : 0;
        
} /* cmp_cpu */



int cmp_page(struct procsortinfo *a, struct procsortinfo *b)
{

    return (a->deltapageflt > b->deltapageflt) ? -1 :
           (a->deltapageflt < b->deltapageflt) ?  1 : 0;
        
} /* cmp_page */



int cmp_res(struct procsortinfo *a, struct procsortinfo *b)
{

    return (a->res > b->res) ? -1 :
           (a->res < b->res) ?  1 : 0;
        
} /* cmp_res */



int cmp_vir(struct procsortinfo *a, struct procsortinfo *b)
{

    return (a->vir > b->vir) ? -1 :
           (a->vir < b->vir) ?  1 : 0;
        
} /* cmp_vir */



/*
 * lets assume that procinfo-array is kept in order by PID
 */
double top_calcsortinfo(
                    struct procinfo *proc1, struct userinfo *user1, int nproc1,
                    struct procinfo *proc2, struct userinfo *user2, int nproc2,
                    struct procsortinfo *procsortinfo)
{
    int i, j;
    double cpusum = 0.0;
    int procmatch;
    struct procinfo *p1, *p2;
    struct userinfo *u1, *u2;
  
    /*
     * lets try finding same process to calculate time process got from cpu
     */
    p1 = proc1; u1 = user1;
    p2 = proc2; u2 = user2;
    /*
     * lets put largest positive long to stop while loop not to go over
     * nproc1 items, and lets hope there are not pid with that number,
     * and something smaller for p2
     */
#if 0
    p1[nproc1].pi_pid = 0x7ffffff;
    p2[nproc2].pi_pid = 0;
    for (i = 0; i < nproc2; i++) { 
	while (p1->pi_pid < p2->pi_pid) {
            p1++;
            u1++;
        }
#else
    j = 0;
    for (i = 0; i < nproc2; i++) { 
	while (p1->pi_pid < p2->pi_pid && j < nproc1) {
            p1++;
            u1++;
            j++;
        }
#endif
	procmatch = (p1->pi_pid == p2->pi_pid);
	
	/*
         * if process is zombie we don't trust the information
         */
	if (p2->pi_stat == SZOMB) {
            procsortinfo[i].cputime = procsortinfo[i].deltacputime = 0.0;
            procsortinfo[i].res = 0;
            procsortinfo[i].vir = 0;
        }
	else {
	    procsortinfo[i].deltacputime =
	    procsortinfo[i].cputime = u2->ui_ru.ru_utime.tv_sec +
		                      u2->ui_ru.ru_utime.tv_usec * 1.0e-6 +
		                      u2->ui_ru.ru_stime.tv_sec +
		                      u2->ui_ru.ru_stime.tv_usec * 1.0e-6;
	    procsortinfo[i].deltapageflt = u2->ui_ru.ru_minflt;
            /* tsize is text size in bytes */
            procsortinfo[i].res = (u2->ui_drss + u2->ui_trss) * 4;
            procsortinfo[i].vir = u2->ui_tsize/1024 + u2->ui_dvm * 4;
	    if (procmatch) { 
		/*
                 * lets watch out for processes which have identical pid 
		 * but which are not same process between two samples
                 */
		if (u1->ui_start == u2->ui_start) {
		    procsortinfo[i].deltacputime -=
                                      (u1->ui_ru.ru_utime.tv_sec +
		                       u1->ui_ru.ru_utime.tv_usec * 1.0e-6 +
		                       u1->ui_ru.ru_stime.tv_sec +
		                       u1->ui_ru.ru_stime.tv_usec * 1.0e-6);
                    procsortinfo[i].deltapageflt -= u1->ui_ru.ru_minflt;
		}
	    }
	}
	procsortinfo[i].index = i;
	/*
	 * in the end cpusum should match refresh_time...
	 */
	cpusum += procsortinfo[i].deltacputime;
	p2++;
        u2++;
    }
    qsort(procsortinfo, nproc2, sizeof(struct procsortinfo), 
          topflag_sort == TOPSORT_CPU  ? cmp_cpu  :
          topflag_sort == TOPSORT_PAGE ? cmp_page :
          topflag_sort == TOPSORT_RES  ? cmp_res  :
          topflag_sort == TOPSORT_VIR  ? cmp_vir  :
                                         cmp_cpu
         );

    return(cpusum);

} /* top_calcsortinfo */



/*
 * if a process entry is invalid then fake that this process is a zombie,
 * so it won't be displayd..., jmaki 921019
 */

#define SWAPPERNAME "swapper"

int top_getprocinfo(struct procinfo *procinfo, struct userinfo *userinfo)
{
    int nproc;
    int i;
    int st;

    nproc = getproc(procinfo, NPROCS, sizeof(struct procinfo));
    for (i = 0; i < nproc; i++) {
        st = getuser(&procinfo[i], sizeof(struct procinfo),
	             &userinfo[i], sizeof(struct userinfo));
        if (st == -1)
            procinfo[i].pi_stat = SZOMB; 
    }
    strcpy(userinfo[0].ui_comm, SWAPPERNAME); /* first process always pid 0 */

    return nproc;

} /* top_getprocinfo */

