/* -- AIX/6000 System monitor 
**
**     get_ifnet.c
**
** Copyright (c) 1991-1995 Jussi Maki, All Rights Reserved.
** Copyright (c) 1993-2001 Marcel Mol, All Rights Reserved.
** NON-COMMERCIAL USE ALLOWED. YOU ARE FREE TO DISTRIBUTE
** THIS PROGRAM AND MODIFY IT AS LONG AS YOU KEEP ORIGINAL
** COPYRIGHTS.
*/

#include <stdio.h>	/* for perror */
#include <stdlib.h>
#include <nlist.h>
#include "config.h"
#include "get_ifnet.h"
#include "getkmemdata.h"

#include <fcntl.h>

#if AIX_VERSION == 3 && (AIX_RELEASE == 1 || AIX_RELEASE == 2)
# include <sys/comio.h>
# include <sys/devinfo.h>
# include <sys/ciouser.h>
# include <sys/entuser.h>
#endif


struct nlist ifnetaddr[] = {
    {"ifnet",   0, 0, 0, 0, 0},
    {NULL,      0, 0, 0, 0, 0},
};

int if_cnt;		/* number of network interfaces */
char *ifnets_names[MAX_IFNETS];

/***************************************************************************
 *                      DATA CAPTURE FUNCTIONS                             *
 ***************************************************************************/


int
get_ifnet(struct ifnet **ifnets)
          /* pointer to array of struct ifnet pointers */
{
    static caddr_t ifnet_addr_first = 0;
    caddr_t ifnet_next;

    if (!ifnet_addr_first) {
        if (knlist(ifnetaddr, 1, sizeof(struct nlist)) == -1)
	    perror("ifnetaddr: entry not found");

        /*
         * get offset for network interfaces
         */
        getkmemdata((char *)&ifnet_addr_first, sizeof(caddr_t),
                    (caddr_t) ifnetaddr[0].n_value);
    }

    ifnet_next = ifnet_addr_first;

    for (if_cnt = 0; ifnet_next; if_cnt++) {
        if (if_cnt >= MAX_IFNETS) {
		/* how can we warn the user... */
            break;
        }

	if (!ifnets[if_cnt]) /* is memory allocated yet */
            ifnets[if_cnt] = (struct ifnet *) calloc(sizeof(struct ifnet), 1);
	getkmemdata((char *) ifnets[if_cnt], sizeof(struct ifnet), ifnet_next);
	ifnet_next = (caddr_t) ifnets[if_cnt]->if_next;
	if (!ifnets_names[if_cnt]) /* is memory allocated yet */
            ifnets_names[if_cnt] = (char *) calloc(MAX_IPNAME, 1);
	getkmemdata(ifnets_names[if_cnt], MAX_IPNAME,
                    (caddr_t) ifnets[if_cnt]->if_name);
	ifnets[if_cnt]->if_name = ifnets_names[if_cnt];

        /*
         * Check if the information is invalid.
         * This could happen in AIX4 since the last if_next
         * pointer is sometimes other that 0x0. Perhaps there is
         * another kernel variable telling the number of interfaces defined.
         * Have to try to guess when the list is in the end.
         *
         * Also according to AIX4 net/netisr.h it seems that the
         * 'struct netisr' is the main linked list which should be used
         * to collect interface information.
         */

        if (ifnets_names[if_cnt][0] < 32 || ifnets_names[if_cnt][0] > 127)
            break;


/*
 * Get fddi statistics via comio routines (define), or
 * via normal kernel structure (undefine).
 * Guess this is needed because fddi bytes_out are not properly 
 * kept in the kernel structure (snmp also has problems with that)
 * (a ptf is availble, guess aix >325 has this fixed...)
 */
/* #undef FDDI_COMIO */
/* #if defined(FDDI_COMIO) */

#if AIX_VERSION == 3 && (AIX_RELEASE == 1 || AIX_RELEASE == 2)

        /*
         * fix the fi0 values by using COMIO to fddi0 
         */
        if (ifnets_names[if_cnt][0] == 'f' && ifnets_names[if_cnt][1] == 'i') 
	    aix32_get_fddi(ifnets[if_cnt]);
#endif
/* #endif */
    }

    return if_cnt;

} /* get_ifnet */



/* #if defined(FDDI_COMIO) */
#if AIX_VERSION == 3 && (AIX_RELEASE == 1 || AIX_RELEASE == 2)
aix32_get_fddi(struct ifnet *ifnet)
{
    cio_stats_t cio;
    char fddiname[64];

    sprintf(fddiname, "fddi%d", ifnet->if_unit);
    get_cio_stats(fddiname, &cio);
    ifnet->if_ipackets = cio.rx_frame_lcnt;
    ifnet->if_opackets = cio.tx_frame_lcnt;
    ifnet->if_ibytes   = cio.rx_byte_lcnt;
    ifnet->if_obytes   = cio.tx_byte_lcnt;

} /* aix32_get_fddi */



get_cio_stats(char *devicename, cio_stats_t *cio_statsp)
{
    int fd;
    static cio_query_blk_t   query_blk;
    static ent_query_stats_t query_stats;
    cio_stats_t *cio;
    ent_stats_t *ds;

    fd = get_comio_device(devicename);
    query_blk.bufptr = (caddr_t)&query_stats;
    query_blk.buflen = sizeof(query_stats);
    query_blk.clearall = 0;
    cio = &query_stats.cc;
    ds = &query_stats.ds;
    if (ioctl(fd, CIO_QUERY, &query_blk) == -1) {
        perror("ioctl CIO_QUERY to device failed");
        exit(1);
    }
    memcpy(cio_statsp, cio, sizeof(cio_stats_t));

    return(0);

} /* get_cio_stats */



#define MAX_COMDEVS 255

int get_comio_device(char *devicename)
{
    static int fds[MAX_COMDEVS];
    static char *fdnames[MAX_COMDEVS];
    static int initted = 0;
    char full_dev_name[128];
    int i;
    int fd;
  
    if (!initted) {
        for (i = 0; i < MAX_COMDEVS; i++) {
            fds[i] = -1;
            fdnames[i] = (char *)0;
        }
        initted=1;
    }
  
    fd = -1;
    for (i = 0; i < MAX_COMDEVS; i++) 
        if (fdnames[i] && strcmp(fdnames[i],devicename) == 0) {
            fd = fds[i];
            break;
        }
        else if (!fdnames[i]) 
            break;
    if (i >= MAX_COMDEVS) {
        printf("get_comio_device(): Error too many communication devices");
        printf("                    Recompile with bigger MAX_COMDEVS in get_ifnet.c");
        exit(1);
    }
    if (fd == -1) {
        strcpy(full_dev_name, "/dev/");
        strcat(full_dev_name, devicename);
        if ((fd = open(full_dev_name, O_RDONLY)) == -1) {
            perror("Opening device");
            printf("name: %s\n", full_dev_name);
            return(-1);
        }
        fdnames[i] = (char *)malloc(strlen(devicename) + 1);
        strcpy(fdnames[i], devicename);
        fds[i] = fd;
    }

    return(fd);

} /* get_comio_device */

#endif

/* #endif */
