/* -- AIX/6000 System monitor 
**
**     get_dkstat.c
**
** Copyright (c) 1991-1995 Jussi Maki, All Rights Reserved.
** Copyright (c) 1993-2001 Marcel Mol, All Rights Reserved.
** NON-COMMERCIAL USE ALLOWED. YOU ARE FREE TO DISTRIBUTE
** THIS PROGRAM AND MODIFY IT AS LONG AS YOU KEEP ORIGINAL
** COPYRIGHTS.
*/

#include <stdio.h>	/* for perror */
#include <stdlib.h>
#include <string.h>	/* for memset */
#include <nlist.h>
#include "config.h"
#include "get_dkstat.h"
#include "getkmemdata.h"
#if defined(DEBUG)
#include "debug.h"
#endif

/* 
 * Calculate disk summary over active disk (define) or 
 * over all disks (undefine)
 */
#undef PER_ACTIVE_DISK

struct nlist diskaddr[] = {
    {"iostat",  0, 0, 0, 0, 0},
    {NULL,      0, 0, 0, 0, 0},
};

int dk_cnt;            /* number of disks in system  */
caddr_t iostat_addr = 0;
struct iostat iostat;


/***************************************************************************
 *                      DATA CAPTURE FUNCTIONS                             *
 ***************************************************************************/

int
init_dkstat()
{

    if (!iostat_addr) {
        if (knlist(diskaddr, 1, sizeof(struct nlist)) == -1)
	    perror("diskaddr: entry not found");
        else
            iostat_addr = (caddr_t) diskaddr[0].n_value;
    }

    getkmemdata((char *) &iostat, sizeof(struct iostat), iostat_addr);
    dk_cnt = iostat.dk_cnt;

    return dk_cnt;
 
} /* init_dkstat */



/*
 * get the dkstat list from kernel. This subroutine will allocate 
 * required memory if list is too short or empty 
 * Usage:
 *        include "get_dkstat.h"
 *
 *        struct dkstat *dkstat;
 *        extern int dk_cnt;
 *
 *        get_dkstat(&dkstat);
 *
 * Return: number of output lines needed...
 */

int
get_dkstat(struct dkstat *dk[])
{
    int i;
    struct dkstat * diskaddr;

#if 0
    /*
     * For now, assume the number of disks does not change
     */
    getkmemdata((char *) &iostat, sizeof(struct iostat), iostat_addr);
    dk_cnt = iostat.dk_cnt;
#endif
 
    for (i = 0; i < dk_cnt; i++) {
	diskaddr = (i == 0 ? (caddr_t) iostat.dkstatp :
                                 (caddr_t) dk[i-1]->dknextp);
        if (diskaddr) {
            if (!dk[i])  {
	        if ((dk[i] = (struct dkstat *) malloc(sizeof(struct dkstat))) == NULL) {
		    break;  /* SHOULD WARN OR QUIT... */
		}
		memset(dk[i], 0, sizeof(struct dkstat));
	    }
            getkmemdata((char *) dk[i], sizeof(struct dkstat), diskaddr);
#if defined(DEBUG)
	    OPENDB;
	    fprintf(debugfp, "disk %d name %s\n", i, dk[i]->diskname);
	    fprintf(debugfp, "disk %d next %0lx\n", i, dk[i]->dknextp);
	    fprintf(debugfp, "disk %d status %0x\n", i, dk[i]->dk_status);
	    fprintf(debugfp, "disk %d time %ld\n", i, dk[i]->dk_time);
	    fprintf(debugfp, "disk %d xrate %ld\n", i, dk[i]->dk_xrate);
	    fprintf(debugfp, "disk %d bsize %ld\n", i, dk[i]->dk_bsize);
	    fprintf(debugfp, "disk %d xfers %ld\n", i, dk[i]->dk_xfers);
	    fprintf(debugfp, "disk %d rblks %ld\n", i, dk[i]->dk_rblks);
	    fprintf(debugfp, "disk %d wblks %ld\n", i, dk[i]->dk_wblks);
	    fprintf(debugfp, "disk %d seek %ld\n", i, dk[i]->dk_seek);
	    CLOSEDB;
#endif
	}
    }
#if defined(DEBUG)
    if (dk[i-1]->dknextp) {
	fprintf(debugfp, "disk %d has followup. Cannot alloc mem or new disks became online (dk_cnt = %d)\n", i-1, dk_cnt);
    }
#endif

    return i;

} /* get_dkstat */



int
summarize_dkstat(struct dkstat *sump,
                     struct dkstat *dk1[], struct dkstat *dk2[])
{
    int i, active = 0;

    memset(sump, 0, sizeof(struct dkstat));
    for (i = 0; i < dk_cnt; i++) {
        sump->dk_bsize += dk1[i]->dk_bsize;
        sump->dk_xrate += dk1[i]->dk_xrate;
        sump->dk_rblks += dk1[i]->dk_rblks - dk2[i]->dk_rblks;
        sump->dk_wblks += dk1[i]->dk_wblks - dk2[i]->dk_wblks;
        sump->dk_xfers += dk1[i]->dk_xfers - dk2[i]->dk_xfers;
        sump->dk_seek  += dk1[i]->dk_seek  - dk2[i]->dk_seek;
        sump->dk_time  += dk1[i]->dk_time  - dk2[i]->dk_time;
        if (dk1[i]->dk_xfers - dk2[i]->dk_xfers)
            active++;
    }

#if defined(PER_ACTIVE_DISK)
    if (active) {
        sump->dk_bsize /= active;
        sump->dk_xrate /= active;
    }
#else
    if (dk_cnt) {
        sump->dk_bsize /= dk_cnt;
        sump->dk_xrate /= dk_cnt;
    }
#endif

    return active;

} /* summarize_dkstat */
