/* -- AIX/6000 System monitor 
**
**     get_cpuinfo.c
**
** Copyright (c) 1993-2001 Marcel Mol, All Rights Reserved.
** Copyright (c) 1991-1995 Jussi Maki, All Rights Reserved.
** NON-COMMERCIAL USE ALLOWED. YOU ARE FREE TO DISTRIBUTE
** THIS PROGRAM AND MODIFY IT AS LONG AS YOU KEEP ORIGINAL
** COPYRIGHTS.
*/

/*
 * AIX 3.1 and AIX 3.2 do not support CPU info ???
 * #if AIX_VERSION != 3 || (AIX_RELEASE != 1 && AIX_RELEASE != 2)
 */
#include "config.h"
#if AIX_NUM  >= 4000000 

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <curses.h>
#include <sys/types.h>
#include <sys/sysinfo.h>
#include <sys/systemcfg.h>
#include <sys/sysconfig.h>
#include <nlist.h>
#include "get_sysvminfo.h"
#include "get_cpuinfo.h"
#include "getkmemdata.h"
#include "print.h"
#if defined(DEBUG)
#include "debug.h"
int cgets = 0, calcs = 0;
#endif

static struct nlist kernelnames[] = {
    {"cpuinfo", 0, 0, 0, 0, 0},
    {NULL,      0, 0, 0, 0, 0},
};


int n_cpus;

extern FILE *logfp;

#ifdef MAIN
main(int argc, char **argv)
{
    int i, ci = 0;
    struct cpuinfo *cpus[2] = {NULL, NULL};

    n_cpus = get_cpuinfo(&cpus[ci]);
    ci ^= 1;
    initscr();
    clear();
    move(0,0);
    printw("RS/6000 SMP System monitor, (c) Jussi Maki, 1996\n");
    printw("Number of CPUs: %d\n", n_cpus);

    while (1) {
        sleep(1);
        get_cpuinfo(&cpus[ci]);
        ci ^= 1;
        print_cpuinfo(1.0, cpus[ci], cpus[ci^1]);
        refresh();
    }

} /* main */
#endif /* MAIN */





int get_cpuinfo(struct cpuinfo **cpu)
{
  
    n_cpus = _system_configuration.ncpus;

    if (!kernelnames[0].n_value) {
        if (knlist(kernelnames, 1, sizeof(struct nlist)) == -1)
	    perror("knlist, entry not found");
    }

    if (!*cpu)
        *cpu = (struct cpuinfo *)calloc(n_cpus, sizeof(struct cpuinfo));

#if defined(DEBUG)
    OPENDB;
    fprintf (debugfp, "get cpu %d, into %0lx\n", ++cgets, *cpu);
    CLOSEDB;
#endif
    getkmemdata((char *) *cpu, sizeof(struct cpuinfo) * n_cpus, 
	        (caddr_t) kernelnames[0].n_value);

    return n_cpus;

} /* get_cpuinfo */



/******************************
  print_cpuinfo.c
 ******************************/

/*
 * Variables to calculate interval/sample values
 */
double c_idle[MAX_CPUS], c_user[MAX_CPUS], c_kern[MAX_CPUS], c_wait[MAX_CPUS];
double c_tidle, c_tuser, c_tkern, c_twait;
double c_pswitch[MAX_CPUS], c_syscall[MAX_CPUS], c_sysread[MAX_CPUS],
       c_syswrite[MAX_CPUS], c_writech[MAX_CPUS], c_readch[MAX_CPUS];
double c_tpswitch, c_tsyscall, c_tsysread,
       c_tsyswrite, c_twritech, c_treadch;

/*
 * Variables to maintain max values
 */
double c_idle_max[MAX_CPUS], c_user_max[MAX_CPUS],
       c_kern_max[MAX_CPUS], c_wait_max[MAX_CPUS];
double c_tidle_max, c_tuser_max, c_tkern_max, c_twait_max;
double c_pswitch_max[MAX_CPUS], c_syscall_max[MAX_CPUS],
       c_sysread_max[MAX_CPUS], c_syswrite_max[MAX_CPUS],
       c_writech_max[MAX_CPUS], c_readch_max[MAX_CPUS];
double c_tpswitch_max, c_tsyscall_max, c_tsysread_max,
       c_tsyswrite_max, c_twritech_max, c_treadch_max;



/*
 * Local function declarations
 */

/*******************************************************************
 * Functions
 *******************************************************************/

#define CPUDELTA(i, a) (cpus1[i].a - cpus2[i].a)
#define CPURATE(i, a)  ((cpus1[i]->a - cpus2[i]->a) / refresh_time)


void
calc_cpuinfo(double refresh_time, struct cpuinfo cpus1[], struct cpuinfo cpus2[])
{
    int i;
    double c_u, c_k, c_w, c_i;
    double c_sum;

    c_tuser = 0;
    c_tkern = 0;
    c_twait = 0;
    c_tidle = 0;
    c_tpswitch  = 0;
    c_tsyscall  = 0;
    c_tsysread  = 0;
    c_tsyswrite = 0;
    c_twritech  = 0;
    c_treadch   = 0;
   
#if defined(DEBUG)
    OPENDB;
    fprintf (debugfp, "calc cpu %d form %0lx and %0lx\n", ++calcs, cpus1, cpus2);
    CLOSEDB;
#endif
    for (i=0; i < n_cpus; i++) {
        c_u = CPUDELTA(i, cpu[CPU_USER]);
        if (c_u < 0) c_u = 0;
        c_k = CPUDELTA(i, cpu[CPU_KERNEL]);
        if (c_k < 0) c_k = 0;
        c_w = CPUDELTA(i, cpu[CPU_WAIT]);
        if (c_w < 0) c_w = 0;
        c_i = CPUDELTA(i, cpu[CPU_IDLE]);
        if (c_i < 0) c_i = 0;
        c_tuser += c_u;
        c_tkern += c_k;
        c_twait += c_w;
        c_tidle += c_i;
        c_sum  = (c_i + c_u + c_k + c_w) / 100.0;
        c_idle[i] = c_i / c_sum;
        c_user[i] = c_u / c_sum;
        c_kern[i] = c_k / c_sum;
        c_wait[i] = c_w / c_sum;
        SETMAX(c_idle[i], c_idle_max[i]);
        SETMAX(c_user[i], c_user_max[i]);
        SETMAX(c_kern[i], c_kern_max[i]);
        SETMAX(c_wait[i], c_wait_max[i]);

        c_i = CPUDELTA(i, pswitch);
        c_tpswitch   += c_i;
        c_pswitch[i]  = c_i / refresh_time;
        SETMAX(c_pswitch[i], c_pswitch_max[i]);
        c_tsyscall   += c_i = CPUDELTA(i, syscall);
        c_syscall[i]  = c_i / refresh_time;
        SETMAX(c_syscall[i], c_syscall_max[i]);
        c_tsysread   += c_i = CPUDELTA(i, sysread);
        c_sysread[i]  = c_i / refresh_time;
        SETMAX(c_sysread[i], c_sysread_max[i]);
        c_tsyswrite  += c_i = CPUDELTA(i, syswrite);
        c_syswrite[i] = c_i / refresh_time;
        SETMAX(c_syswrite[i], c_syswrite_max[i]);
        c_twritech   += c_i = CPUDELTA(i, writech);
        c_writech[i]  = c_i / refresh_time;
        SETMAX(c_writech[i], c_writech_max[i]);
        c_treadch    += c_i = CPUDELTA(i, readch);
        c_readch[i]   = c_i / refresh_time;
        SETMAX(c_readch[i], c_readch_max[i]);
    }
    c_sum  = (c_tidle + c_tuser + c_tkern + c_twait) / 100;
    c_tuser /= c_sum;
    c_tkern /= c_sum;
    c_twait /= c_sum;
    c_tidle /= c_sum;
    SETMAX(c_tidle, c_tidle_max);
    SETMAX(c_tuser, c_tuser_max);
    SETMAX(c_tkern, c_tkern_max);
    SETMAX(c_twait, c_twait_max);

    c_tpswitch  /= refresh_time;
    c_tsyscall  /= refresh_time;
    c_tsysread  /= refresh_time;
    c_tsyswrite /= refresh_time;
    c_twritech  /= refresh_time;
    c_treadch   /= refresh_time;
    SETMAX(c_tpswitch,  c_tpswitch_max);
    SETMAX(c_tsyscall,  c_tsyscall_max);
    SETMAX(c_tsysread,  c_tsysread_max);
    SETMAX(c_tsyswrite, c_tsyswrite_max);
    SETMAX(c_twritech,  c_twritech_max);
    SETMAX(c_treadch,   c_treadch_max);

    return;

} /* calc_cpuinfo */



void
print_cpuinfo_init()
{
    int x, y;

    x = 0; y = 5;
    move(y + 0, x);
    BOLDON;
    printw("CPU USER KERN WAIT IDLE%% PSW SYSCALL WRITE  READ WRITEkb  READkb");
    BOLDOFF;

    return;

} /* print_cpuinfo_init */



void
print_cpuinfo()
{
    int i;
    int x, y;

#define myprw1(s, a) if (a < 100000) printw(" %*d", s, (int) a ); \
                  else printw(" %*dk", s-1, (int) a / 1000);

#define MB  1048576
#define KB  1024
#define myprw2(a) if ((a) < 10000) printw(" %7.2f", (a) ); \
                  else if ((a) < 100000*KB) printw(" %6dk", (int) (a) / KB); \
                  else printw(" %6dM", (int) (a) / MB);

    x = 0; y = 5;
    /*
     * Per CPU
     */
    for (i = 0; i < n_cpus; i++) {
        move(y + 1 + i, x);
        printw("#%-2d %4d %4d %4d %4d",
	     i, (int) c_user[i], (int) c_kern[i],
                (int) c_wait[i], (int) c_idle[i]); 

        myprw1(5, c_pswitch[i]);
        myprw1(6, c_syscall[i]);
        myprw1(5, c_syswrite[i]);
        myprw1(5, c_sysread[i]);

	myprw2(c_writech[i] / 1024);
	myprw2(c_readch[i]  / 1024);
    }
    /*
     * SUM
     */
    move(y + 1 + i, x);
    printw("SUM %4d %4d %4d %4d",
	   (int) c_tuser, (int) c_tkern, (int) c_twait, (int) c_tidle);
    myprw1(5, c_tpswitch);
    myprw1(6, c_tsyscall);
    myprw1(5, c_tsyswrite);
    myprw1(5, c_tsysread);
    myprw2(c_twritech / 1024);
    myprw2(c_treadch  / 1024);

    return;

} /* print_cpuinfo */



void
log_cpuinfo_header()
{

    fprintf(logfp, "# smp[_max] cpunumber sys%% wait%% user%% idle%%"
                   " pswitch syscall sysread syswrite readch writech\n");
    fprintf(logfp, "#    pswitch    process_switches/second\n");
    fprintf(logfp, "#    syscall    total_system_calls/second\n");
    fprintf(logfp, "#    sysread    read_syscalls/second\n");
    fprintf(logfp, "#    syswrite   write_syscalls/second\n");
    fprintf(logfp, "#    readch     characters_read_by_read_syscall/second\n");
    fprintf(logfp, "#    writech    characters_written_by_write_syscall/second\n");

    return;

} /* log_cpuinfo_header */



void
log_cpuinfo()
{
    int i;

    for (i = 0; i < n_cpus; i++) {
        fprintf(logfp,
            "smp cpu%03d %.1f %.1f %.1f %.1f %.1f %.1f %.1f %.1f %.1f %.1f\n",
	     i, c_kern[i], c_wait[i],    c_user[i],    c_idle[i], 
	     c_pswitch[i], c_syscall[i], c_sysread[i], c_syswrite[i],
	     c_readch[i],  c_writech[i]);
    }
    fprintf(logfp,
            "smp cpuall %.1f %.1f %.1f %.1f %.1f %.1f %.1f %.1f %.1f %.1f\n",
	   c_tkern,    c_twait,    c_tuser,    c_tidle,
	   c_tpswitch, c_tsyscall, c_tsysread, c_tsyswrite,
	   c_treadch,  c_twritech);

    return;

} /* log_cpuinfo */



void
max_cpuinfo_init()
{
    int i;

    for (i = 0; i < n_cpus; i++) {
        c_idle_max[i] = 0;
        c_user_max[i] = 0;
        c_kern_max[i] = 0;
        c_wait_max[i] = 0;
        c_pswitch_max[i]  = 0;
        c_syscall_max[i]  = 0;
        c_sysread_max[i]  = 0;
        c_syswrite_max[i] = 0;
        c_writech_max[i]  = 0;
        c_readch_max[i]   = 0;
    }

    c_tidle_max = 0;
    c_tuser_max = 0;
    c_tkern_max = 0;
    c_twait_max = 0;
    c_tpswitch_max  = 0;
    c_tsyscall_max  = 0;
    c_tsysread_max  = 0;
    c_tsyswrite_max = 0;
    c_twritech_max  = 0;
    c_treadch_max   = 0;

    return;

} /* max_cpuinfo_init */


void
log_cpuinfo_max()
{
    int i;

    for (i = 0; i < n_cpus; i++) {
        fprintf(logfp,
          "smp_max cpu%03d %.1f %.1f %.1f %.1f %.1f %.1f %.1f %.1f %.1f %.1f\n",
	     i, c_kern_max[i], c_wait_max[i],
             c_user_max[i],    c_idle_max[i], 
	     c_pswitch_max[i], c_syscall_max[i],
             c_sysread_max[i], c_syswrite_max[i],
	     c_readch_max[i],  c_writech_max[i]);
    }
    fprintf(logfp,
           "smp_max cpuall %.1f %.1f %.1f %.1f %.1f %.1f %.1f %.1f %.1f %.1f\n",
	   c_tkern_max,    c_twait_max,    c_tuser_max,    c_tidle_max,
	   c_tpswitch_max, c_tsyscall_max, c_tsysread_max, c_tsyswrite_max,
	   c_treadch_max,  c_twritech_max);

    return;

} /* log_cpuinfo_max */


#else /* AIX_NUM > 3020000 */

void
calc_cpuinfo(double refresh_time, struct cpuinfo *cpus1, struct cpuinfo *cpus2)
{
    return;
} /* calc_cpuinfo */


void
print_cpuinfo_init()
{
    return;
} /* print_cpuinfo_init */


void
print_cpuinfo()
{
    return;
} /* print_cpuinfo */


void
log_cpuinfo_header()
{
    return;
} /* log_cpuinfo_header */

void
log_cpuinfo()
{
    return;
} /* log_cpuinfo */


void max_cpuinfo_init()
{
    return;
} /* max_cpuinfo_init */


void
log_cpuinfo_max()
{
    return;
} /* log_cpuinfo_max */


int get_cpuinfo(struct cpuinfo **cpu)
{
    return 0;
}

#endif /* AIX_NOM > 03020000 */
