/* -- AIX/6000 System monitor 
**
**     filesys.c
**
** Copyright (c) 1993-2001 Marcel Mol, All Rights Reserved.
** Copyright (c) 1991-1995 Jussi Maki, All Rights Reserved.
** NON-COMMERCIAL USE ALLOWED. YOU ARE FREE TO DISTRIBUTE
** THIS PROGRAM AND MODIFY IT AS LONG AS YOU KEEP ORIGINAL
** COPYRIGHTS.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/statfs.h>
#include <sys/mntctl.h>
#include <sys/vmount.h>
#include <fstab.h>
#include <errno.h>
#include "config.h"

#define mntent	vmount			/* mount structure */

extern int logallfs;

/* ******************************************************************** */
/*                      general program constants                       */
/* ******************************************************************** */
#define	STRLENGTH	80		/* length of a string */
#define MAXFS           500		/* maximum number of file systems */
#define FSSTATLEN	96000
#define MB		1048576		/* 1Mb = 1048576 Bytes */

	/* macros used to access the strings through the use of the index */
#define FsName(i)	fs[i].fs_name
#define DevName(i)	fs[i].dev_name
#define MntTyp(i)	fs[i].mnt_typ
#define	MntOpt(i)	fs[i].mnt_opt
#define FsStat(i)	fs[i].stats

/* ******************************************************************** */
/*                       program data structures                        */
/* ******************************************************************** */

/* *** string to hold file system name *** */
typedef char String[STRLENGTH];

/* *** struct to hold all the string info copied form static system areas *** */
typedef struct {
	String        fs_name,	/* FS names */
                      dev_name,	/* Device Names */
                      mnt_typ,	/* type of device mount */
                      mnt_opt;	/* option of device mount */
        struct statfs stats;	/* file system stats */
} fsinfo;

fsinfo fs[MAXFS];
int    NumFS;
#if defined(MAIN)
FILE * logfp = stdout;
#else
extern FILE * logfp;
#endif

/*
 * Function declarations
 */
void df_print (int fsnum);
void print_fs_header();
void dump_fs(int getfs);
void get_fs ();



#if defined(MAIN)
main()
{
    int i;

    get_fs();
    printf("%-15s %-15s %-8s %-8s %-8s %-6s %s\n",
     "FsName", "DevName", "Tot MB", "Free MB", "Avail MB", "Type", "Options");
    for (i = 0; i < NumFS; i++)
        df_print(i);
    dump_fs_info();

}



/*
 * Write out the detailed info       NOT USED IN MONITOR
 */
void
df_print(int fsnum)
{
    long     bsize      = FsStat(fsnum).f_bsize, 
             blocks     = FsStat(fsnum).f_blocks,
             bfree      = FsStat(fsnum).f_bfree, 
             bavail     = FsStat(fsnum).f_bavail;

    printf("%-15s %-15s", FsName(fsnum), DevName(fsnum));
    /*
     * total size
     */
    (blocks == -1 || bsize == -1) ? printf (" Undef   ") : printf (" %8.2f", 
                          (((double)(bsize)/(float)(MB)))*(double)(blocks));
    /*
     * space available for root
     */
    (bfree == -1 || bsize == -1) ? printf (" Undef   ")  : printf (" %8.2f",
                           (((double)(bsize)/(float)(MB)))*(double)(bfree));

    /*
     * space available for users
     */
    (bavail == -1 || bsize == -1) ? printf (" Undef   ") : printf (" %8.2f",
                          (((double)(bsize)/(float)(MB)))*(double)(bavail));

    printf(" %-6s", MntTyp(fsnum));
    printf(" %s\n", MntOpt(fsnum));

    return;

} /* df_print */
#endif



/*
 * Dump record format to logfile
 */
void
print_fs_header()
{

    fprintf(logfp, "# filesys name device type options blocksize"
                   " blocks freeblocks availblocks indodes freeinodes\n");  

    return;

} /* print_fs_header */



/*
 * Dump fs info to logfile
 */
void
dump_fs(int getfs)
{
    int i;

    if (getfs)
        get_fs();

    for (i = 0; i < NumFS; i++)
        fprintf(logfp, "filesys %s %s %s %s %ld %ld %ld %ld %ld %ld\n",
               FsName(i), DevName(i), MntTyp(i), MntOpt(i), FsStat(i).f_bsize,
               FsStat(i).f_blocks, FsStat(i).f_bfree, FsStat(i).f_bavail,
               FsStat(i).f_files, FsStat(i).f_ffree);

    return;

} /* dump_fs */


/*
 * Get_fs get the file systems which are mounted
 */
void
get_fs ()
{
    char            buffer[FSSTATLEN];
    int             i, j;
    struct mntent * mnt_info = (struct vmount *)buffer;

    if ((NumFS = mntctl(MCTL_QUERY, sizeof(buffer), buffer)) == 0) {
        fprintf(logfp, "# error: buffer too small, recompile filesys.c with "
                       "FSSTATLEN set to %d\n", *((int *) buffer));
        return;
    }

    if (NumFS > MAXFS) {
        fprintf(logfp, "# error: too many filesystems: recompile filesys.c"
                 " with MAXFS > %d. Ignoring surplus...\n", MAXFS);
        NumFS = MAXFS;
    }

    if (NumFS < 1) {
        fprintf(logfp, "# error: mount returned no valid file systems\n");
        return;
    }


    for (i = 0, j = 0; j < NumFS; j++) {
        if (mnt_info->vmt_gfstype != MNT_NFS || logallfs) {
            strcpy (FsName(i), vmt2datasize(mnt_info, VMT_STUB) ?
                               vmt2dataptr(mnt_info, VMT_STUB)  : "N.A.");
            strcpy (DevName(i), vmt2datasize(mnt_info, VMT_OBJECT) ?
                                vmt2dataptr(mnt_info, VMT_OBJECT)  : "N.A.");

            /* *** of course IBM had to make things difficult *** */
            switch (mnt_info -> vmt_gfstype) {
                case MNT_AIX: strcpy (MntTyp(i), "aix");
                              break;
                /*
                 * PTD addition - we define AFS to be type 4 and treat
                 *                it like NFS
                 */
                case 4      : strcpy (MntTyp(i), "afs");
                              break;
                case MNT_NFS: strcpy (MntTyp(i), "nfs");
                              break;
                case MNT_JFS: strcpy (MntTyp(i), "jfs");
                              break;
                case MNT_CDROM: strcpy (MntTyp(i), "cdrom");
                              break;
                default: strcpy (MntTyp(i), "user");
                              break;
            }
            strcpy (MntOpt(i), "");
            strcat (MntOpt(i), (mnt_info -> vmt_flags & MNT_READONLY) ?
                                      "ro" : "rw");
            if (mnt_info -> vmt_flags & MNT_REMOVABLE)
                strcat (MntOpt(i), ",removable");
            if (mnt_info -> vmt_flags & MNT_DEVICE)
                strcat (MntOpt(i), ",device");
            if (mnt_info -> vmt_flags & MNT_REMOTE)
                strcat (MntOpt(i), ",remote");
            if (mnt_info -> vmt_flags & MNT_UNMOUNTING)
                strcat (MntOpt(i), ",unmounting");
            if (mnt_info -> vmt_flags & MNT_SYSV_MOUNT)
                strcat (MntOpt(i), ",sysv_mount");
            if (mnt_info -> vmt_flags & MNT_NOSUID)
                strcat (MntOpt(i), ",nosuid");
            if (mnt_info -> vmt_flags & MNT_NODEV)
                strcat (MntOpt(i), ",no_device_access");

            /*
             * read file system status
             */
            if (statfs (FsName(i), &(FsStat(i))) == -1)
                fprintf(logfp, "# error: %s: %s\n", FsName(i), strerror(errno));

            i++;
        }

        mnt_info = (struct vmount *)((char *) mnt_info + mnt_info->vmt_length);

    }

    /*
     * Reset NumFS to the actual number of filesystems we saved.
     */
    NumFS = i;

    return;

} /* get_fs */
